function [contacts]= calculate_contacts(pdb, seqName, sequence)

% Retrieve the selected chain in the structure
chain_pos =  cell2mat(regexp(seqName, '_'));
chain = '';
if(chain_pos<length(seqName))
    chain = seqName(chain_pos+1:end);
end
if(~isempty(chain))
    if(chain == '.')
        chain = 'A';
    elseif(all(chain=='ad'))
        chain = '3';    
    elseif(all(chain=='ac'))
        chain = '2';  
    end
else
    chain = 'A';
end

% Read PDB file
pdb_model = pdb.Model.Atom;
pdb_model = pdb_model([pdb_model.chainID]==chain);

% Parameters
r_probe = 2.8;
lim_dist_ca = 22.5;
contacts = [];

% Select atom names
atom_name = {pdb_model.AtomName};

% Select all possible combination of CA atoms between AA
ca_pos = ismember(atom_name,'CA');
pdb_ca = pdb_model(ca_pos);
positions = [pdb_ca.resSeq];
aa_pairs = nchoosek(positions,2);
CA_coor = [[pdb_ca.X]' [pdb_ca.Y]' [pdb_ca.Z]'];
D = pdist(CA_coor)';
pos_distances = (aa_pairs(:,2) - aa_pairs(:,1))>=5 & D<lim_dist_ca;
aa_pairs = aa_pairs(pos_distances,:);

pos_neg = regexp(atom_name,'^C$|^CA$|^N$|^O$|^OXT$|^H*');
pos_neg = cellfun(@isempty,pos_neg);
pdb_filtered = pdb_model(pos_neg);

% Filter aminoacids without the atom restriction
filtered_atoms = ismember(aa_pairs(:,1),[pdb_filtered.resSeq]) & ismember(aa_pairs(:,2),[pdb_filtered.resSeq]);
aa_pairs = aa_pairs(filtered_atoms,:);

% Calculate atoms distances between each pair of aminoacids
for i=1:size(aa_pairs,1)
    pair = aa_pairs(i,:);
    aa1 = pdb_filtered([pdb_filtered.resSeq]==pair(1));
    aa2 = pdb_filtered([pdb_filtered.resSeq]==pair(2));
    atoms1 = {aa1.AtomName};
    atoms2 = {aa2.AtomName};
    
    [m1 m2] = meshgrid(1:length(atoms1), 1:length(atoms2));
    
    if(length(atoms1)==1)
        atoms1 = atoms1(m1(:));
    else
        atoms1 = atoms1(m1(:))';
    end

    if(length(atoms2)==1)
        atoms2 = atoms2(m2(:));
    else
        atoms2 = atoms2(m2(:))';
    end
    
    atoms_pairs = [atoms1 atoms2];
    
    % Measure distance between atoms
    [nu,pos1] = ismember(atoms_pairs(:,1),{aa1.AtomName});
    [nu,pos2] = ismember(atoms_pairs(:,2),{aa2.AtomName});
    coor1 = [[aa1(pos1).X]' [aa1(pos1).Y]' [aa1(pos1).Z]'];
    coor2 = [[aa2(pos2).X]' [aa2(pos2).Y]' [aa2(pos2).Z]'];
    dist = pdist2(coor1,coor2);
    dist = diag(dist);    
    lim = radius(atoms_pairs(:,1), unique({aa1.resName})) + radius(atoms_pairs(:,2), unique({aa2.resName})) + r_probe;
    
    % If any pair of atoms is close enough, 
    % both AAs are considered in contact.
    if(any(dist<lim))
        contacts = [contacts; aa_pairs(i,1) aa_pairs(i,2)];
    end   
end
%fclose(fileID);

% Fix contacts positions according to the real analyzed sequence
pdbseq = pdb.Sequence([pdb.Sequence.ChainID]==chain).Sequence;
% Shift contacts if the real sequence is partially included
[nonused, contacts] = ismember(contacts,positions');
% Remove contacts in final and initial positions not included in the analyzed sequence
contacts = contacts(contacts(:,1) > 0 & contacts(:,2) > 0,:);
contacts = contacts(contacts(:,1) < length(sequence) & contacts(:,2) < length(sequence),:);
